package com.example

import okhttp3.*
import org.apache.http.HttpHost
import org.apache.http.auth.AuthScope
import org.apache.http.auth.UsernamePasswordCredentials
import org.apache.http.client.CredentialsProvider
import org.apache.http.client.methods.HttpGet
import org.apache.http.impl.client.BasicCredentialsProvider
import org.apache.http.impl.client.CloseableHttpClient
import org.apache.http.impl.client.HttpClients
import org.apache.http.util.EntityUtils
import java.io.IOException
import java.net.InetSocketAddress
import java.net.Proxy
import java.nio.charset.StandardCharsets


fun main(args: Array<String>) {
	val ip = "192.168.10.10" // 代理主机地址
	val port = 50000 // 代理主机端口

	// 使用OKHttp库
	OKHttpProxy.build(ip, port).test()
//	 OKHttpProxy.build(ip, port, "aaaaaa", "bbbbbb").test(); // 代理认证

	// 使用HttpClient库
//	 HttpClientProxy.build(ip, port).test();
//	 HttpClientProxy.build(ip, port, "aaaaaa", "bbbbbb").test(); // 代理认证
}

/**
 * OKHttp库使用代理
 */
internal class OKHttpProxy {
	var proxyHost: String? = null
	var proxyPort = 0
	var proxyAccount: String? = null
	var proxyPwd: String? = null

	@Throws(IOException::class)
	fun test() {
		val targetUrl = "http://myip.ipip.net"
		var client: OkHttpClient? = null
		client = if (proxyAccount == null || proxyPwd == null) {
			getHttpClient(proxyHost, proxyPort)
		} else {
			// 账号密码验证
			getHttpClient(proxyHost, proxyPort, proxyAccount!!, proxyPwd!!)
		}
		val request: Request = Request.Builder()
			.url(targetUrl)
			.build()
		val response: Response = client.newCall(request).execute()
		println(response.body?.string())
	}

	companion object {
		/**
		 * @param host 代理主机地址
		 * @param port 代理主机端口
		 */
		fun build(host: String?, port: Int): OKHttpProxy {
			val proxy = OKHttpProxy()
			proxy.proxyHost = host
			proxy.proxyPort = port
			return proxy
		}

		/**
		 * @param host 代理主机地址
		 * @param port 代理主机端口
		 * @param acc 代理认证账号
		 * @param pwd 代理认证口令
		 */
		fun build(host: String?, port: Int, acc: String?, pwd: String?): OKHttpProxy {
			val proxy = OKHttpProxy()
			proxy.proxyHost = host
			proxy.proxyPort = port
			proxy.proxyAccount = acc
			proxy.proxyPwd = pwd
			return proxy
		}

		/**
		 * 代理不需要账号密码认证的httpClient
		 */
		private fun getHttpClient(proxyHost: String?, proxyPort: Int): OkHttpClient {
			val proxy = Proxy(Proxy.Type.HTTP, InetSocketAddress(proxyHost, proxyPort))
			return OkHttpClient.Builder()
				.proxy(proxy)
				.build()
		}

		/**
		 * 代理需要账号密码认证的httpClient
		 */
		private fun getHttpClient(proxyHost: String?, proxyPort: Int, acc: String, pwd: String): OkHttpClient {
			val proxy = Proxy(Proxy.Type.HTTP, InetSocketAddress(proxyHost, proxyPort))
			// 账号密码验证
			val authenticator: Authenticator = Authenticator { _, response ->
				val credential: String = Credentials.basic(acc, pwd)
				response.request.newBuilder().header("Proxy-Authorization", credential).build()
			}
			return OkHttpClient.Builder()
				.proxy(proxy)
				.proxyAuthenticator(authenticator)
				.build()
		}
	}
}

/**
 * HttpClient库使用代理
 */
internal class HttpClientProxy {
	var proxyHost: String? = null
	var proxyPort = 0
	var proxyAccount: String? = null
	var proxyPwd: String? = null

	@Throws(IOException::class)
	fun test() {
		val targetUrl = "http://myip.ipip.net"
		var client: CloseableHttpClient? = null
		client = if (proxyAccount == null || proxyPwd == null) {
			getHttpClient(proxyHost, proxyPort)
		} else {
			// 账号密码验证
			getHttpClient(proxyHost, proxyPort, proxyAccount!!, proxyPwd!!)
		}
		val httpGet = HttpGet(targetUrl)
		val response = client.execute(httpGet)
		val resultStr: String = EntityUtils.toString(response.entity, StandardCharsets.UTF_8)
		println(resultStr)
	}

	companion object {
		/**
		 * @param host 代理主机地址
		 * @param port 代理主机端口
		 */
		fun build(host: String?, port: Int): HttpClientProxy {
			val proxy = HttpClientProxy()
			proxy.proxyHost = host
			proxy.proxyPort = port
			return proxy
		}

		/**
		 * @param host 代理主机地址
		 * @param port 代理主机端口
		 * @param acc 代理认证账号
		 * @param pwd 代理认证口令
		 */
		fun build(host: String?, port: Int, acc: String?, pwd: String?): HttpClientProxy {
			val proxy = HttpClientProxy()
			proxy.proxyHost = host
			proxy.proxyPort = port
			proxy.proxyAccount = acc
			proxy.proxyPwd = pwd
			return proxy
		}

		/**
		 * 代理不需要账号密码认证的httpClient
		 */
		private fun getHttpClient(proxyHost: String?, proxyPort: Int): CloseableHttpClient {
			val proxy = HttpHost(proxyHost, proxyPort, "HTTP")
			return HttpClients.custom()
				.setProxy(proxy)
				.build()
		}

		/**
		 * 代理需要账号密码认证的httpClient
		 */
		private fun getHttpClient(proxyHost: String?, proxyPort: Int, acc: String, pwd: String): CloseableHttpClient {
			val proxy = HttpHost(proxyHost, proxyPort, "HTTP")
			val provider: CredentialsProvider = BasicCredentialsProvider()
			provider.setCredentials(AuthScope(proxy), UsernamePasswordCredentials(acc, pwd))
			return HttpClients.custom()
				.setProxy(proxy)
				.setDefaultCredentialsProvider(provider)
				.build()
		}
	}
}
